Direct Debit Payment Initiation
===============================

* [Sample usage of DirectDebit File](#sample-usage-of-directdebit-file)
* [Sample Usage DirectDebit with Factory](#sample-usage-of-directdebit-with-facade-factory)
* [Sample Usage DirectDebit with Factory and Custom Header](#sample-usage-directdebit-with-factory-and-custom-header)
* [Add an amendment to a transfer](#add-an-amendment-to-a-transfer)
* [Add address information to transaction](#add-address-information-to-transaction)


Sample usage of DirectDebit File
-------------------------------------
The following example creates a DirectDebit file, adds a PaymentInformation Object and a single transaction to it.  
The variable names are used to describe what should be contained within them.

```php
    use Digitick\Sepa\GroupHeader;
    use Digitick\Sepa\PaymentInformation;
    use Digitick\Sepa\DomBuilder\DomBuilderFactory;
    use Digitick\Sepa\TransferFile\CustomerDirectDebitTransferFile;
    use Digitick\Sepa\TransferInformation\CustomerDirectDebitTransferInformation;

    $groupHeader = new GroupHeader($messageIdentification, $companyName);
    $transferFile = new CustomerDirectDebitTransferFile($groupHeader);

    $paymentInfo = new PaymentInformation(
        $paymentInfoId,
        $originAccIban,
        $originAccBic,
        $companyName
    );
    $paymentInfo->setCreditorId($creditorId);
    $paymentInfo->setSequenceType(PaymentInformation::S_ONEOFF);
    $paymentInfo->setOriginName($companyName);

    $transfer = new CustomerDirectDebitTransferInformation(1234, $clientAccIban, $clientName, $endToEndId);
    $transfer->setBic($clientAccBic);
    $transfer->setRemittanceInformation($clientLabel);
    $transfer->setMandateId($mandateId);
    $transfer->setMandateSignDate(new DateTime());
    $transfer->setFinalCollectionDate(new DateTimeImmutable());

    $transfer->setCountry('BG');
    $transfer->setPostCode('1000');
    $transfer->setTownName('Nowhere');
    $transfer->setStreetName('Some Street');
    $transfer->setBuildingNumber(12);
    $transfer->setFloorNumber(13);
    //Retrieve auto-generated UUIDv4 and store it:
    $autoGeneratedUuid = $transfer->getUUID();
    //You can add as many transfers as you want to a single PaymentInformation object
    $paymentInfo->addTransfer($transfer);
    //You can add as many PaymentInformation objects as you want to a transfer file
    $transferFile->addPaymentInformation($paymentInfo);

    // Write to a file:
    $domBuilder = DomBuilderFactory::createDomBuilder($transferFile, $painFormat); //For e.g. 'pain.008.001.09'
    file_put_contents($filePath, $domBuilder->asXml());
    // ...or retrieve the \DomDocument object, modify it and do something else with it:
    $domBuilder->asDoc();
```

Sample Usage of DirectDebit with Facade Factory
-------------------------------------
The library also provides facades to achieve the same thing. In this example we let the facade create the GroupHeader for us:

```php
use Digitick\Sepa\TransferFile\Factory\TransferFileFacadeFactory;
use Digitick\Sepa\PaymentInformation;

// Returns a CustomerDirectDebitFacade
$directDebit = TransferFileFacadeFactory::createDirectDebit('SampleUniqueMsgId', 'SampleInitiatingPartyName', 'pain.008.001.09');

// Add a PaymentInfo object (PmtInf), it's possible to create multiple such objects in one ISO20022 file
// "firstPayment" is the identifier for the transactions aka the PaymentInfoId (PmtInfId)
// This creates a once-off Direct Debit. If needed you can use ::S_FIRST, ::S_RECURRING or ::S_FINAL respectively. Note that the sequence type is set 
// on the PaymentInfo level meaning that it is one and the same for a group of transfers.
$directDebit->addPaymentInfo('firstPayment', array(
    'id'                    => 'firstPayment',
    'dueDate'               => new DateTimeImmutable('now + 7 days'), // Optional. If not passed 'now' is used
    'creditorName'          => 'My Company',
    'creditorAccountIBAN'   => 'FI1350001540000056',
    'creditorAgentBIC'      => 'PSSTFRPPMON',
    'seqType'               => PaymentInformation::S_ONEOFF,
    'creditorId'            => 'DE21WVM1234567890', //This is your company's SEPA CreditorIdentifier. It's very important
    'localInstrumentCode'   => 'CORE',
    // Add/Set batch booking option as per your requirement, optional
    'batchBooking'          => true, 
));

// Add a Single Transaction to the named PaymentInfo
$directDebit->addTransfer('firstPayment', array(
    'amount'                => 500, // `amount` in cents
    'debtorIban'            => 'FI1350001540000056',
    'debtorBic'             => 'OKOYFIHH',
    'debtorName'            => 'Their Company',
    'debtorMandate'         => 'AB12345',
    'debtorMandateSignDate' => '13.10.2012',
    'remittanceInformation' => 'Purpose of this direct debit',
    // Optional but quite important. This identifier travels full-circle through the banking system.
    // For e.g. if you were to receive a payment reject for this payment this is one of the ways to concretely identify it.
    // Since version 2.3.0 the library also supports the UETR element which is essentially an UUIDv4
    'endToEndId'            => 'MyUniqueClutchId',
    // What the Customer actually sees in their statement
    'remittanceInformation' => 'Order 123456'
));
// Retrieve the resulting XML
$directDebit->asXML();
```

Sample Usage DirectDebit with Factory and Custom Header
-------------------------------------------------------
In this example we pass our own custom GroupHeader:

```php
use Digitick\Sepa\TransferFile\Factory\TransferFileFacadeFactory;
use Digitick\Sepa\PaymentInformation;
use Digitick\Sepa\GroupHeader;

//Set the custom header (Spanish banks example) information
$header = new GroupHeader(date('Y-m-d-H-i-s'), 'Me');
$header->setInitiatingPartyId('DE21WVM1234567890');

$directDebit = TransferFileFacadeFactory::createDirectDebitWithGroupHeader($header, 'pain.008.001.09');

$directDebit->addPaymentInfo('firstPayment', array(
    'id'                    => 'firstPayment',
    'dueDate'               => new DateTime('now + 7 days'), // optional. Otherwise default period is used
    'creditorName'          => 'My Company',
    'creditorAccountIBAN'   => 'FI1350001540000056',
    'creditorAgentBIC'      => 'PSSTFRPPMON',
    'seqType'               => PaymentInformation::S_ONEOFF,
    'creditorId'            => 'DE21WVM1234567890',
    'localInstrumentCode'   => 'CORE' // default. optional.
));

$directDebit->addTransfer('firstPayment', array(
    'amount'                => 500,
    'debtorIban'            => 'FI1350001540000056',
    'debtorBic'             => 'OKOYFIHH',
    'debtorName'            => 'Their Company',
    'debtorMandate'         => 'AB12345',
    'debtorMandateSignDate' => '13.10.2012',
    'remittanceInformation' => 'Order 123456',
    'endToEndId'            => 'MyUniqueClutchId',
    
));
// Retrieve the resulting XML
$directDebit->asXML();
```

Add an amendment to a transfer
------------------------------
Add a Single Transaction to the named PaymentInfo object
```php
$directDebit->addTransfer('firstPayment', array(
    'amount'                  => 500,
    'debtorIban'              => 'FI1350001540000056',
    'debtorBic'               => 'OKOYFIHH',
    'debtorName'              => 'Their Company',
    'remittanceInformation'   => 'Purpose of this credit transfer',
    'endToEndId'              => 'Invoice-No X' // optional, if you want to provide additional structured info
    // Amendments start here
    'originalMandateId'       => '1234567890',
    'originalDebtorIban'      => 'AT711100015440033700',
    'amendedDebtorAccount'    => true
));
```

Add address information to transaction
--------------------------------------
If the debtor account belongs to a bank that is not a member of the European Economic Area (EEA), the address data of the account holder must be added to the transaction.
For sure one must do this of the following countries: Switzerland, Andorra, Monaco, San Marino, Vatican City and the United Kingdom.
Through it is generally a good practice to add this data anyway. 


```php
$directDebit->addTransfer('firstPayment', [
    'amount'            => 1499,
    'debtorIban'        => 'CH6089144731137988786',
    'debtorBic'         => 'CRESCHZZXXX',
    'debtorName'        => 'John Doe',
    // ...
    // and the relevant address data
    'debtorCountry'     => 'CH',
    'postCode'          => '8245',
    'townName'          => 'Feuerthalen',
    'streetName'        => 'Example Street',
    'buildingNumber'    => '12',
    'floorNumber'       => '13'
]);
````
