<?php

namespace webmail\mail;

use core\exception\InvalidStateException;
use webmail\service\EmailService;
use core\Context;
use webmail\model\Email;
use core\exception\ResourceException;
use webmail\service\ConnectorService;
use webmail\solr\SolrMail;
use webmail\solr\SolrMailQuery;
use webmail\mail\action\MailActionsBase;
use webmail\service\CloudTokenService;

class SendMail {
    
    protected $to = array();
    protected $cc = array();
    protected $bcc = array();
    
    protected $extraHeaders = array();
    
    protected $subject = '';
    protected $fromName = null;
    protected $fromEmail = null;
    
    protected $attachmentFiles = array();
    protected $attachmentDataFiles = array();
    
    protected $emailId = null;
    protected $identityId = null;
    
    protected $content = null;
    
    protected $error = null;
    
    public function __construct() {
        $ctx = object_container_get(Context::class);
        $this->setFromEmail( $ctx->getCompanyEmail() );
        
    }
    
    public function addTo($email, $name=null) {
        $this->to[] = array('name' => $name, 'email' => $email);
    }
    public function getTo() { return $this->to; }
    public function clearTo() { $this->to = array(); }
    
    public function addCc($email, $name=null) {
        $this->cc[] = array('name' => $name, 'email' => $email);
    }
    public function getCc() { return $this->to; }
    public function clearCc() { $this->to = array(); }
    
    public function addBcc($email, $name=null) {
        $this->bcc[] = array('name' => $name, 'email' => $email);
    }
    public function getBcc() { return $this->to; }
    public function clearBcc() { $this->to = array(); }
    
    public function addHeader($name, $val) {
        $this->extraHeaders[] = array(
            'name' => $name,
            'value' => $val
        );
    }
    
    public function setEmailId($id) { $this->emailId = $id; }
    public function getEmailId() { return $this->emailId; }
    
    public function setIdentityId($id) { $this->identityId = $id; }
    public function getIdentityId() { return $this->identityId; }
    
    public function getFromName() { return $this->fromName; }
    public function setFromName($n) { $this->fromName = $n; }
    
    public function getFromEmail() { return $this->fromEmail; }
    public function setFromEmail($e) { $this->fromEmail = $e; }
    
    public function getSubject() { return $this->subject; }
    public function setSubject($s) { $this->subject = $s; }
    
    public function setContent($c) { $this->content = $c; }
    public function getContent() { return $this->content; }
    
    public function getError() { return $this->error; }
    
    
    public function addAttachmentFile($file, $filename=null) {
        $this->attachmentFiles[] = array(
            'file' => $file,
            'filename' => $filename
        );
    }
    
    public function addAttachmentDataFile($data, $filename, $opts=array()) {
        $df = array(
            'data' => $data,
            'filename' => $filename
        );
        
        if (isset($opts['id'])) {
            $df['id'] = $opts['id'];
        }
        if (isset($opts['disposition'])) {
            $df['disposition'] = $opts['disposition'];
        }
        if (isset($opts['content-type'])) {
            $df['content-type'] = $opts['content-type'];
        }
        
        $this->attachmentDataFiles[] = $df;
    }
    
    public function addAttachmentInline($data, $filename, $id, $contentType) {
        $df = array(
            'id' => $id,
            'content-type' => $contentType,
            'disposition' => 'inline',
            'data' => $data,
            'filename' => $filename
        );
        
        
        $this->attachmentDataFiles[] = $df;
        
    }
    
    
    
    
    public function buildMessage() {
        $message = new \Swift_Message( $this->getSubject() );
        
        foreach($this->extraHeaders as $h) {
            $message->getHeaders()->addTextHeader( $h['name'], $h['value'] );
        }
        
        $message->setFrom(array($this->getFromEmail() => $this->getFromName()));
        
        foreach($this->to as $m) {
            $message->addTo($m['email'], $m['name']);
        }
        foreach($this->cc as $m) {
            $message->addCc($m['email'], $m['name']);
        }
        foreach($this->bcc as $m) {
            $message->addBcc($m['email'], $m['name']);
        }
        
        $message->setBody($this->getContent(), 'text/html', 'UTF-8');
        
        foreach($this->attachmentFiles as $f) {
            $data = file_get_contents($f['file']);
            
            if ($data === false) {
                throw new InvalidStateException('Attachment not found');
            }
            
            if (!$f['filename'])
                $f['filename'] = basename($f['file']);
                
                $att = new \Swift_Attachment($data, $f['filename']);
                $message->attach($att);
        }
        
        foreach($this->attachmentDataFiles as $f) {
            $att = new \Swift_Attachment($f['data'], $f['filename']);
            if (isset($f['id']))
                $att->setId( $f['id']);
            
            if (isset($f['disposition']))
                $att->setDisposition($f['disposition']);
            if (isset($f['content-type']))
                $att->setContentType($f['content-type']);
            
            $message->attach($att);
        }
        
        return $message;
    }
    
    
    public function send() {
        /** @var EmailService $emailService */
        $emailService = object_container_get(EmailService::class);
        $settings = $emailService->getMailServerSettings();
        
        // debug mode? => force local
        if (is_debug()) {
            if ( $settings['server_type'] != 'local' && $settings['server_type'] != 'local_mail' )
                $settings['server_type'] = 'local';
        }
        
        if ($settings['server_type'] == 'local') {
            // hmz..
            if (defined('SMTP_HOST') && SMTP_HOST) {
                $transport = new \Swift_SmtpTransport(SMTP_HOST, SMTP_PORT);
                if (defined('SMTP_USERNAME') && defined('SMTP_PASSWORD') && SMTP_USERNAME && SMTP_PASSWORD) {
                    $transport->setUsername(SMTP_USERNAME);
                    $transport->setPassword(SMTP_PASSWORD);
                }
            } else {
                // unix? => use sendmail
                $transport = @new \Swift_SendmailTransport();
                
                
                // skip check, just try.. some environments block this, but still work
//                 if (file_exists('/usr/sbin/sendmail')) {
//                     $transport = new \Swift_SendmailTransport();
//                 } else {
//                     throw new ResourceException('/usr/sbin/sendmail not found');
//                 }
            }
        } else if ($settings['server_type'] == 'local_mail') {
            $transport = new \Swift_MailTransport( );
        } else if ($settings['server_type'] == 'azure') {
            $ctService = object_container_get( CloudTokenService::class );
            $wat = $ctService->readAzureToken( $settings['azure_token_id'] );
            
            $token = $ctService->getAzureAccessToken( $wat->getWebmailAzureTokenId() );
            if (!$token) {
                $message = t('Azure access token not set');
                $this->error = $message;
                ctx()->setLastError( $message );
                return false;
            }
            
            $transport = new \Swift_SmtpTransport( 'smtp.office365.com', 587, 'tls' );
            $transport->setAuthMode( 'XOAUTH2' );
            
            $transport->setUsername( $wat->getAzureSmtpUsername() );
            $transport->setPassword( $token );
            
        } else if ($settings['server_type'] == 'smtp') {
            // TLS security for transport?
            $transport_security = '';
            if (isset($settings['mail_tls']) && $settings['mail_tls']) {
                $transport_security = 'tls';
            }
            
            $transport = new \Swift_SmtpTransport($settings['mail_hostname'], $settings['mail_port'], $transport_security);
            if ($settings['mail_username'] && $settings['mail_password']) {
                $transport->setUsername($settings['mail_username']);
                $transport->setPassword($settings['mail_password']);
            }
        }
        else {
            throw new InvalidStateException( 'Unknown servertype' );
        }
        
        $message = @$this->buildMessage();
//         $message->setEncoder( new \Swift_Mime_ContentEncoder_PlainContentEncoder('8bit') );
        
        $mailer = new \Swift_Mailer($transport);
        
//         $mailLogger = new \Swift_Plugins_Loggers_ArrayLogger();
//         $mailer->registerPlugin(new \Swift_Plugins_LoggerPlugin($mailLogger));
        
        
        $ctx = object_container_get(Context::class);
        if ($ctx->getContextName() == 'demo') {
            // don't send mail in demo-environment
            return true;
        } else {
//             $r = $mailer->send( $message );
//             print $mailLogger->dump();exit;
            
            try {
                $r = @$mailer->send( $message );
            } catch (\Exception $ex) {
                $this->error = $ex->getMessage();
                ctx()->setLastError( $ex->getMessage() );
                return false;
            }
            
            // mail sent & identity is set? => try to save mail on imap server (if configured)
            if ($r && $this->getIdentityId()) {
                // call SolrMailActions::saveSendMail()
                // might return false if imap is not configured
                $mailActions = MailActionsBase::getInstance();
                $mailActions->saveSendMail( $this );
            }
            
            // mail successfully sent?
            if ($r) {
                if ($this->getEmailId()) {
                    $emailService->markMailAsSent( $this->getEmailId() );
                }
                
            }
            
            return $r;
        }
    }
    
    
    public static function createMail(Email $email) {
        $ctx = object_container_get(Context::class);
        
        $sm = new SendMail();
        $sm->setEmailId($email->getEmailId());
        $sm->setIdentityId( $email->getIdentityId() );
        $sm->setSubject($email->getSubject());
        $sm->setFromName($email->getFromName());
        $sm->setFromEmail($email->getFromEmail());
        
        // add In-Reply-To header if available
        if ($email->getSolrMailId()) {
            $solrMail = SolrMailQuery::readStaticById( $email->getSolrMailId() );
            if ($solrMail && $solrMail->getEmlMessageId())
                $sm->addHeader('In-Reply-To', $solrMail->getEmlMessageId());
        }
        
        
        foreach($email->getRecipients() as $r) {
            if (strtolower($r->getToType()) == 'cc') {
                $sm->addCc($r->getToEmail(), $r->getToName());
            } else if (strtolower($r->getToType()) == 'bcc') {
                $sm->addBcc($r->getToEmail(), $r->getToName());
            }
            // default to 'To'
            else {
                $sm->addTo($r->getToEmail(), $r->getToName());
            }
        }
        $sm->setContent($email->getTextContent());
        
        foreach($email->getFiles() as $f) {
            $full_path = $ctx->getDataDir() . '/' . $f->getPath();
            $data = file_get_contents($full_path);
            
            if ($data === false) {
                throw new InvalidStateException('Attachment not found');
            }
            
            $sm->addAttachmentDataFile($data, $f->getFilename());
        }
        
        return $sm;
    }
    
    
    
}
