<?php



use core\Context;
use core\exception\FileException;
use core\exception\NotForLiveException;
use core\exception\InvalidStateException;
use core\db\DatabaseHandler;

function is_get() {
    return $_SERVER['REQUEST_METHOD'] == 'GET';
}

function is_post() {
    return $_SERVER['REQUEST_METHOD'] == 'POST';
}

function is_cli() {
    return php_sapi_name() == 'cli';
}

function is_web() {
    return php_sapi_name() != 'cli';
}

function is_windows() {
    if (strpos(PHP_OS, 'WIN') === 0) {
        return true;
    }
    
    return false;
}

/**
 * toolbox_get_class() - get_class(), but if toolbox-proxy is used, return name parent object
 * @param $obj
 * @return string
 */
function toolbox_get_class( $obj ) {
    
    if ($obj instanceof \core\container\ObjectHookProxyInterface) {
        $ro = new \ReflectionObject($obj);
        $n = $ro->getParentClass()->getName();
    }
    else {
        $n = get_class( $obj );
    }
    
    return $n;
}



/**
 * Internet Explorer-browser? (not Edge)
 * @return boolean
 */
function browser_ie() {
    if (isset($_SERVER['HTTP_USER_AGENT']) && strpos(@$_SERVER['HTTP_USER_AGENT'], 'Trident') !== false) {
        return true;
    }
    
    return false;
}

function is_debug() {
    if (defined('DEBUG') && DEBUG) {
        return true;
    } else {
        return false;
    }
}

/** @return \core\Context */
function ctx() {
    return \core\Context::getInstance();
}


function is_admin_context() {
    if (defined('ADMIN_CONTEXT') && ADMIN_CONTEXT == true) {
        return true;
    } else {
        return false;
    }
}



function is_true($v) {
    if ($v) {
        return true;
    }
    if ($v == 'true') {
        return true;
    }
    
    return false;
}

function is_false($v) {
    if (!$v) {
        return true;
    }
    if ($v == 'false') {
        return true;
    }
    
    return false;
}


function get_class_shortname( $obj ) {
    $n = toolbox_get_class($obj);
    
    $p = strrpos($n, '\\');
    if ($p !== false) {
        $n = substr($n, $p+1);
    }
    
    return $n;
}




function appUrl($u) {
    
    if (defined('ADMIN_CONTEXT')) {
        $contextName = 'admin';
    } else {
        $contextName = Context::getInstance()->getContextName();
    }
    
    if (strpos($u, ':') !== false) {
        $protocol = substr($u, 0, strpos($u, ':'));
        $protocol = strtolower( $protocol );
        if (in_array($protocol, array('javascript', 'http', 'https', 'mailto', 'tel')))
            return $u;
    }
    
    // /module/-path? => rewrite to /?mpf=/module/.../
    if (strpos($u, '/module/') === 0) {
        $urlParts = explode('/', $u);
        
        // module_path found? (module exists..)
        if (module_exists( $urlParts[2] )) {
            $u = '/?mpf='.$u;
        }
    }
    
    if (strpos($u, BASE_HREF . $contextName . '/') === 0) {
        return $u;
    }
    
    if (is_standalone_installation()) {
        if (defined_value('REWRITE_DISABLED') && strpos($u, '/module/') === 0) {
            $url = BASE_HREF . '?mpf=' . substr($u, strlen(BASE_HREF)-1);
        }
        else {
            $url = BASE_HREF . substr($u, 1);
        }
        
    } else {
        $url = BASE_HREF . $contextName . $u;
    }
    
    $url = apply_filter('appUrl', $url);
    
    return $url;
}



function buildAppUrl( $url, $params ) {
    $urlFromRequest = false;
    
    if ($url === null) {
        $url = $_SERVER['REQUEST_URI'];
        
        $urlFromRequest = true;
    }
    
    list ($path, $urlparams) = $url;
    
    if (strpos($url, '?') !== false) {
        list ($path, $urlparams) = explode('?', $url, 2);
        
        $tokens = explode('&', $urlparams);
        foreach($tokens as $t) {
            list( $k, $v ) = explode('=', $t, 2);
            
            $k = urldecode($k);
            $v = urldecode($v);
            
            if (isset($params[$k]) == false)
                $params[$k] = $v;
        }
        
    }
    
    if ($urlFromRequest)
        $r = $path;
    else
        $r = appUrl( $path );
    
    if (count($params)) {
        $r .= '?' . http_build_query($params);
    }
    
    return $r;
}

function buildUrl( $url, $params ) {
    if ($url === null) {
        $url = $_SERVER['REQUEST_URI'];
    }
    
    list ($path, $urlparams) = $url;
    
    if (strpos($url, '?') !== false) {
        list ($path, $urlparams) = explode('?', $url, 2);
        
        $tokens = explode('&', $urlparams);
        foreach($tokens as $t) {
            list( $k, $v ) = explode('=', $t, 2);
            
            $k = urldecode($k);
            $v = urldecode($v);
            
            if (isset($params[$k]) == false)
                $params[$k] = $v;
        }
        
    }
    
    $r = $path;
    if (count($params)) {
        $r .= '?' . http_build_query($params);
    }
    
    return $r;
}




function jsUrl( $moduleName, $publicPath ) {
    
    $publicPath = trim($publicPath);
    if (strpos($publicPath, '/') === 0)
        $publicPath = substr($publicPath, 1);
    
    $f = module_file( $moduleName, '/public/'.$publicPath );
    if (file_exists( $f ) == false) {
        return null;
    }
    
    $p = 'mpf='.urlencode('/module/'.$moduleName.'/'.$publicPath).'&v='.filemtime($f);
    
    $u = appUrl('/?'.$p);
    
    return $u;
}



/**
 * app_request_uri() - returns (relative) request_uri. Filters BASE_HREF & user-contextName
 */
function app_request_uri() {
    if (is_standalone_installation()) {
        return '/'.substr($_SERVER['REQUEST_URI'], strlen(BASE_HREF));
    } else {
        $uri = '/'.substr($_SERVER['REQUEST_URI'], strlen(BASE_HREF));
        
        // '/module/' paths doesn't contain contextNames
        if (strpos($uri, '/module/') === 0)
            return $uri;
        
        $matches = array();
        if (preg_match('/^\\/([a-zA-Z0-9_-]+)?\\/.*/', $uri, $matches) == false || count($matches) != 2) {
            throw new \core\exception\ContextNotFoundException('context not found');
        }
        
        $uri = substr($uri, strlen($matches[1])+1);
        return $uri;
    }
}

function request_uri_no_params() {
    if (is_cli()) {
        throw new InvalidStateException('request_uri_no_params() called in cli-env');
    }
    
    $uri = $_SERVER['REQUEST_URI'];
    $p = strrpos($uri, '?');
    
    if ($p !== false) {
        $uri = substr($uri, 0, $p);
    }
    
    return urldecode($uri);
}


function redirect($url) {
    
    
    if ( DatabaseHandler::getInstance()->inTransaction() ) {
        throw new InvalidStateException( 'redirect() called but in a database Transaction' );
    }
    
    // TODO: check if $url is prefixed? & clean up?
    header('Location: ' . appUrl($url));
    exit;
}

function remote_addr() {
    
    // wrapper for support of proxy's in the future
    
    return $_SERVER['REMOTE_ADDR'];
}

function define_true($name) {
    if (defined($name) == false) {
        define($name, true);
    } else if (constant($name) !== true) {
        throw new InvalidStateException($name.' already defined: '.var_export(constant($name), true));
    }
}

/**
 * defined_value() - returns value, or $defaultValue, false by default
 */
function defined_value($name, $defaultValue = false) {
    if (defined($name)) {
        return constant( $name );
    }
    else {
        return $defaultValue;
    }
}

function isset_value( &$p, $defaultValue = false ) {
    if (isset($p)) {
        return $p;
    }
    else {
        return $defaultValue;
    }
}



function list_files($path, $opts=array()) {
    $path = realpath( $path );
    
    if (!$path)
        return false;
    
    if (isset($opts['basepath']) == false)
        $opts['basepath'] = $path;
    
    $dh = opendir( $path );
    if (!$dh) return false;
    
    $files = array();
    
    while ($f = readdir($dh)) {
        if ($f == '.' || $f == '..') continue;
        
        if (isset($opts['dironly']) && $opts['dironly'] && is_dir($path.'/'.$f) == false) {
            continue;
        }

        $skipFile = false;
        if (isset($opts['fileonly']) && $opts['fileonly'] && is_dir($path.'/'.$f) == true) {
            $skipFile = true;
        }
        
        if ($skipFile == false) {
            if (isset($opts['append-slash']) && $opts['append-slash'] && is_dir( $path . '/' . $f )) {
                $files[] = $f . '/';
            } else {
                $files[] = $f;
            }
        }
        
        
        if (isset($opts['recursive']) && $opts['recursive'] && is_dir($path.'/'.$f)) {
            $subfiles = list_files($path . '/' . $f, $opts);
            
            for($x=0; $x < count($subfiles); $x++) {
                $subfile = $subfiles[$x];
                
                if (strpos($subfiles[$x], $opts['basepath']) !== false)
                    $subfile = substr($subfile, strlen($opts['basepath']));
                
                $subfile = $f . '/' . $subfiles[$x];
                
                $subfiles[$x] = $subfile;
            }
            
            $files = array_merge($files, $subfiles);
        }
        
    }
    
    return $files;
}

function file_extension($file) {
    $p = strrpos($file, '.');
    
    if ($p === false) {
        return false;
    }
    
    $ext = substr($file, $p+1);
    $ext = strtolower($ext);
    
    return $ext;
}

function file_mime_type($file) {
    
    $mime_types = array(
        'txt' => 'text/plain',
        'htm' => 'text/html',
        'html' => 'text/html',
        'php' => 'text/html',
        'css' => 'text/css',
        'less' => 'text/css',
        'js' => 'application/javascript',
        'json' => 'application/json',
        'xml' => 'application/xml',
        'swf' => 'application/x-shockwave-flash',
        'flv' => 'video/x-flv',
        
        // images
        'png' => 'image/png',
        'jpe' => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'jpg' => 'image/jpeg',
        'gif' => 'image/gif',
        'bmp' => 'image/bmp',
        'ico' => 'image/vnd.microsoft.icon',
        'tiff' => 'image/tiff',
        'tif' => 'image/tiff',
        'svg' => 'image/svg+xml',
        'svgz' => 'image/svg+xml',
        
        // archives
        'zip' => 'application/zip',
        'rar' => 'application/x-rar-compressed',
        'exe' => 'application/x-msdownload',
        'msi' => 'application/x-msdownload',
        'cab' => 'application/vnd.ms-cab-compressed',
        
        // audio/video
        'mp3' => 'audio/mpeg',
        'qt' => 'video/quicktime',
        'mov' => 'video/quicktime',
        
        // adobe
        'pdf' => 'application/pdf',
        'psd' => 'image/vnd.adobe.photoshop',
        'ai' => 'application/postscript',
        'eps' => 'application/postscript',
        'ps' => 'application/postscript',
        
        // ms office
        'doc' => 'application/msword',
        'rtf' => 'application/rtf',
        'xls' => 'application/vnd.ms-excel',
        'ppt' => 'application/vnd.ms-powerpoint',
        
        // open office
        'odt' => 'application/vnd.oasis.opendocument.text',
        'ods' => 'application/vnd.oasis.opendocument.spreadsheet',
    );
    
    $ext = file_extension($file);
    
    if (isset($mime_types[$ext])) {
        return $mime_types[$ext];
    }
    
    return 'application/octet-stream';
}



function load_php_file($file) {
    $r = include $file;
    return $r;
}


function get_var($key, $defaultVal=null) {
    $ctx = Context::getInstance();
    
    return $ctx->getVar($key, $defaultVal);
}

function set_request_var( $key, $val ) {
    $_REQUEST[$key] = $val;
    $_GET[$key]     = $val;
    $_POST[$key]    = $val;
}


function has_file($paramName) {
    if (isset($_FILES[$paramName]) && isset($_FILES[$paramName]['size']) && $_FILES[$paramName]['size'] > 0) {
        return true;
    }
    
    return false;
}

function save_upload_to($paramFile, $pathInDatadir) {
    $filename = basename( $_FILES[$paramFile]['name'] );
    
    $path = $pathInDatadir . '/' . $filename;
    $path = preg_replace('/\\/+/', '/', $path);
    $path = preg_replace('/\\\\+/', '/', $path);
    
    return save_upload($paramFile, $path);
}
function save_upload($paramFile, $path) {
    $ctx = Context::getInstance();
    
    $datadirContext = realpath($ctx->getDataDir());
    if ($datadirContext == false)
        throw new FileException('DATA_DIR doesn\'t exist');
    
    if (file_exists($datadirContext) == false) {
        if (mkdir($datadirContext, 0755, true) == false) {
            throw new FileException('Unable to create DATA_DIR for ' . $ctx->getContextName());
        }
    }
    
    $datadirContext = realpath($datadirContext);

    // shouldn't/cant happen
    if ($datadirContext == false) {
        throw new FileException('Data directory not found for ' . $ctx->getContextName());
    }
    
    $fullpath = $datadirContext . '/' . $path;
    $filename = basename($path);
    
    $dir = dirname( $fullpath);
    if (file_exists($dir) == false) {
        if (!mkdir($dir, 0755, true))
            return false;
    }
    
    // check if dir is within contextName's path
    $dir = realpath($dir);
    if (strpos($dir, $datadirContext) !== 0) {
        throw new FileException('Accessing file out of DATA_DIR for context (1)');
    }
    
    $fullpath = $dir . '/' . $filename;
    if (strpos($fullpath, $datadirContext) !== 0) {
        throw new FileException('Accessing file out of DATA_DIR for context (2)');
    }
    
    $r = copy($_FILES[$paramFile]['tmp_name'], $fullpath);
    
    if ($r) {
        return substr(realpath( $fullpath ), strlen($datadirContext)+1);
    } else {
        return false;
    }
}

function save_data($file, $data) {
    $ctx = Context::getInstance();
    
    $datadirContext = realpath($ctx->getDataDir());
    if ($datadirContext == false)
        throw new FileException('DATA_DIR doesn\'t exist');
        
    if (file_exists($datadirContext) == false) {
        if (mkdir($datadirContext, 0755, true) == false) {
            throw new FileException('Unable to create DATA_DIR for ' . $ctx->getContextName());
        }
    }
    
    $datadirContext = realpath($datadirContext);
    
    // shouldn't/cant happen
    if ($datadirContext == false) {
        throw new FileException('Data directory not found for ' . $ctx->getContextName());
    }
    
    $fullpath = $datadirContext . '/' . $file;
    $filename = basename($file);
    
    $dir = dirname( $fullpath );
    if (file_exists($dir) == false) {
        if (!mkdir($dir, 0755, true))
            return false;
    }
    
    // check if dir is within contextName's path
    $dir = realpath($dir);
    if (strpos($dir, $datadirContext) !== 0) {
        throw new FileException('Accessing file out of DATA_DIR for context (1)');
    }
    
    $fullpath = $dir . '/' . $filename;
    if (strpos($fullpath, $datadirContext) !== 0) {
        throw new FileException('Accessing file out of DATA_DIR for context (2)');
    }
    
    $r = file_put_contents($fullpath, $data);
    
    if ($r !== false) {
        return substr(realpath( $fullpath ), strlen($datadirContext)+1);
    } else {
        return false;
    }
}


function delete_data_file($f) {
    $file = get_data_file($f);
    
    if ($file === false) {
        return false;
    }
    
    return unlink($file);
}

function delete_data_path($path, $recursive = false) {
    $fullpath = get_data_file($path);
    
    if ($fullpath === false)
        return false;
    
    if (is_file($fullpath)) {
        return unlink($fullpath);
    } else if (is_dir($fullpath)) {
        if ($recursive) {
            $subfiles = list_data_directory( $path );
            
            foreach($subfiles as $subfile) {
                delete_data_path( $path . '/' . $subfile, true );
            }
        }
        
        return rmdir( $fullpath );
    }
}

/**
 * lists only files (not directories)
 */
function list_data_files($pathInDataDir) {
    $ctx = Context::getInstance();
    
    $datadirContext = realpath($ctx->getDataDir());
    if ($datadirContext == false)
        return array();
        
    $dir = realpath( $datadirContext . '/' . $pathInDataDir );
    if (strpos($dir, $datadirContext) !== 0)
        return array();
    
    $dh = opendir($dir);
    $files = array();
    while ($f = readdir($dh)) {
        if (is_file($dir . '/' . $f)) {
            $files[] = $f;
        }
    }
    closedir($dh);
    
    return $files;
}

/**
 * lists all files + directories
 */
function list_data_directory($pathInDataDir) {
    $ctx = Context::getInstance();
    
    $datadirContext = realpath($ctx->getDataDir());
    if ($datadirContext == false)
        return array();
        
        $dir = realpath( $datadirContext . '/' . $pathInDataDir );
    if (strpos($dir, $datadirContext) !== 0)
        return array();
    
    $dh = opendir($dir);
    $files = array();
    while ($f = readdir($dh)) {
        if ($f == '.' || $f == '..') continue;
        
        $files[] = $f;
    }
    closedir($dh);
    
    return $files;
}


function get_data_file_safe($basePath, $subPath) {
    // get fullpath for base
    $basePath = get_data_file( $basePath );
    if (!$basePath) {
        return false;
    }
    
    $fullpath = realpath( $basePath . '/' . $subPath);
    if ($fullpath == false) {
        return false;
    }
    
    if (strpos($fullpath, $basePath) === 0) {
        return $fullpath;
    }
    
    return false;
}

function get_data_file($f) {
    $ctx = Context::getInstance();

    $datadirContext = realpath($ctx->getDataDir());
    if ($datadirContext == false)
        return false;
    
    $dir = realpath( dirname($datadirContext . '/' . $f) );
    if (strpos($dir, $datadirContext) !== 0)
        return false;
    
    $file = realpath($datadirContext . '/' . $f);
    if ($file == false)
        return false;
    
    if (strpos($file, $datadirContext) !== 0)
        return false;
    
    return $file;
}

function get_data_bytes($f) {
    $file = get_data_file($f);
    if (!$file) {
        return false;
    }
    
    return file_get_contents( $file );
}


function copy_data_tmp($file, $tmpname=null) {
    // create temp-folder
    $tmpfolder = get_data_file('/tmp');
    if ($tmpfolder == false) {
        $ctx = Context::getInstance();
        $f = $ctx->getDataDir();
        
        if (mkdir($f . '/tmp', 0755) == false) {
            throw new FileException('Unable to create temp-folder');
        }
        
        $tmpfolder = get_data_file('/tmp');
    }
    
    if ($tmpfolder === false || file_exists($tmpfolder) == false) {
        throw new FileException('Temp-folder not found');
    }
    
    
    // no name specified? => generate temp-file name
    $dest = null;
    if ($tmpname === null) {
        for($x=0; $x < 50; $x++) {
            $f = 'temp'.date('Ymd').rand(0, 999999999).'-'.basename($file);
            
            if (file_exists($tmpfolder . '/' . $f) == false) {
                $dest = $tmpfolder . '/' . $f;
                break;
            }
        }
        
        if ($dest === null) {
            throw new FileException('Unable to determine temp-filename');
        }
    }
    // tempname specified
    else {
        $dest = $tmpfolder . '/' . basename($tmpname);
    }
    
    if (copy($file, $dest)) {
        return $dest;
    } else {
        throw new FileException('Unable to copy file to temp-folder');
    }
}


function url_data_file($f) {
    if (get_data_file($f)) {
        return appUrl('/?m=core&c=file&f='.urlencode($f));
    }
    
    return '';
}


function format_filesize($bytes) {
    $bytes = (int)$bytes;
    
    if ($bytes <= 1024) {
        $t = $bytes . ' bytes';
    } else if ($bytes <= 1024 * 1024) {
        $t = myround($bytes/1024, 2) . ' kb';
    } else if ($bytes <= 1024 * 1024 * 1024) {
        $t = myround($bytes/(1024 * 1024), 2) . ' mb';
    } else {//if ($bytes <= 1024 * 1024 * 1024 * 1024) {
        $t = myround($bytes/(1024 * 1024 * 1024), 2) . ' gb';
    }
    
    $t = str_replace('.', ',', $t);
    
    return $t;
}



function array_remove_value($arr, $val) {
    $newArr = array();
    
    foreach($arr as $a) {
        if ($a == $val) continue;
        $newArr[] = $a;
    }
    
    return $newArr;
}

function pos_in_array($needle, $array) {
    foreach($array as $key => $val) {
        if ($needle == $val)
            return $key;
    }
    
    return false;
}


function in_iarray($needle, $haystack) {
    $needle = strtolower($needle);
    
    foreach($haystack as $k) {
        if ($k == $needle) {
            return true;
        }
    }
    
    return false;
}


/**
 * check_array() - checks if $variable[..] is an array and contains min-count elements
 */
function check_array($var, $field=null, $minCount=0) {
    if (is_array($var) == false)
        return false;
    
    if ($field != null) {
        if (isset($var[$field]) == false || is_array($var[$field]) == false) {
            return false;
        }
        
        $var = $var[$field];
    }
    
    if (count($var) < $minCount)
        return false;
    
    return true;
}

/**
 * get_array_value() - returns value by name, ie 'arr[0][value]'
 */
function get_array_value( $name, $array, $defaultValue=null ) {
    $v = $array;
    
    $counter=0;
    $tokens = explode( '[', $name );
    foreach($tokens as $t) {
        $t = rtrim($t, ']');
        
        if (isset($v[$t])) {
            $counter++;
            $v = $v[$t];
        }
        else {
            break;
        }
    }
    
    if ($counter == count($tokens)) {
        return $v;
    }
    else {
        return $defaultValue;
    }
}



function lookupModuleFile($pathInModule) {
    $ctx = Context::getInstance();
    
    $basepath = realpath(ROOT . '/modules');
    foreach($ctx->getEnabledModules() as $module) {
        
        $f = realpath($basepath . '/' . $module . '/' . $pathInModule);
        if ($f)
            return $f;
    }
    
    return false;
}




function date2unix($input)
{
    if ($input === null) {
        return null;
    }
    
    $input = trim($input);

    if (strpos($input, '/Date(') !== false) {
        $matches = array();
        
        if (preg_match('/\/Date\((\\d+)\\)\\//', $input, $matches) && count($matches) == 2) {
            return intval($matches[1] / 1000);
        }
    }
    
    if (preg_match('/^\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}/', $input)) {
        try {
            $dt = new DateTime( $input );
            $dt->setTimezone( new DateTimeZone( date_default_timezone_get() ) );
            $input = $dt->format('Y-m-d H:i:s');
        } catch (\Exception $ex) { }
    }

    if (preg_match('/^\\d{8}+$/', $input)) {
        $d2uy = (int)substr($input, 0, 4);
        $d2um = (int)substr($input, 4, 2);
        $d2ud = (int)substr($input, 6, 2);

        if ($d2uy > 1850 && $d2uy < 2500 && $d2um >= 1 && $d2um <= 12 && $d2ud >= 1 && $d2ud <= 31) {
            return mktime(12, 0, 0, $d2um, $d2ud, $d2uy);
        }
    }


    
    if ($input == "0000-00-00") { // ongeldige datum
        return null;
    } else if (preg_match('/^\\d{4}-\\d{1,2}-\\d{1,2} \\d{2}:\\d{2}:\\d{2}$/', $input)) { // jaar-maand-dag uur:minuut:seconde
        list ($date, $time) = explode(' ', $input);
        $d = explode('-', $date);
        $t = explode(':', $time);
    } else if (preg_match('/^\\d{1,2}-\\d{1,2}-\\d{4} \\d{2}:\\d{2}:\\d{2}$/', $input)) { // dag-maand-jaar uur:minuut:seconde
        list ($date, $time) = explode(' ', $input);

        $d = explode("-", $input);
        $d = array(
            $d[2],
            $d[1],
            $d[0]
        );

        $t = explode(':', $time);
    } else if (preg_match('/^\\d{1,2}-\\d{1,2}-\\d{4} \\d{2}:\\d{2}$/', $input)) { // dag-maand-jaar uur:minuut
        list ($date, $time) = explode(' ', $input);

        $d = explode("-", $date);
        $d = array(
            $d[2],
            $d[1],
            $d[0]
        );

        $t = explode(':', $time);
        $t[] = 0;
    } else if (preg_match("/^\\d{1,2}-\\d{1,2}-\\d{4}$/", $input)) {
        $d = explode("-", $input);
        $d = array(
            $d[2],
            $d[1],
            $d[0]
        );
    } else if (preg_match('/^\\d{4}\\/\\d{1,2}\\/\\d{1,2}$/', $input)) {
        $d = explode("/", $input);
    } else if (preg_match('/^\\d{4}-\\d{1,2}-\\d{1,2}$/', $input)) {
        $d = explode("-", $input);
    } else {
        return null;
    }

    if (isset($t)) {
        $t = explode(":", substr($input, 10));
        if (count($t) == 2) $t[] = 0;

        return mktime($t[0], $t[1], $t[2], $d[1], $d[2], $d[0]);
    }

    return mktime(0, 0, 0, $d[1], $d[2], $d[0]);
}


function format_date($str, $format='d-m-Y', $defaultVal='') {
    
    if (!$str)
        return $defaultVal;

    if (strpos($str, 'Y-m') !== false || strpos($str, 'm-Y') !== false || $str == 'Ymd') {
        if (is_debug()) {
            throw new NotForLiveException('Lul, je draait de $str & $format weer eens om...');
        }
        else {
            throw new InvalidArgumentException('$str and $format flipped');
        }
    }
    
    if (valid_date($str) || valid_datetime($str)) {
        $t = date2unix($str);
        
        if ($t) {
            return date($format, $t);
        }
    }
    
    return $defaultVal;
}

function date2number($date) {
    $t = date2unix($date);
    
    return (int)date('Ymd', $t);
}


function dayname2number( $name ) {
    // already number given?
    if (is_numeric($name) && $name >= 1 && $name <= 7) {
        return $name;
    }
    
    $name = strtolower($name);
    
    switch( $name ) {
        case 'monday' :
        case 'maandag' :
        case 'ma' :
        case 'mo' :
            return 1;
        case 'tuesday' :
        case 'dinsdag' :
        case 'di' :
        case 'tu' :
            return 2;
        case 'wednesday' :
        case 'woensdag' :
        case 'wo' :
        case 'we' :
            return 3;
        case 'thursday' :
        case 'donderdag' :
        case 'do' :
        case 'th' :
            return 4;
        case 'friday' :
        case 'vrijdag' :
        case 'vr' :
        case 'fr' :
            return 5;
        case 'saturday' :
        case 'zaterdag' :
        case 'za' :
        case 'sa' :
            return 6;
        case 'sunday' :
        case 'zondag' :
        case 'zo' :
        case 'su' :
            return 7;
    }
    
    return -1;
}


function format_datetime($str, $format='d-m-Y H:i:s', $defaultVal='') {
    
    if (valid_datetime($str)) {
        $t = date2unix($str);

        if ($t) {
            return date($format, $t);
        }
    }
    
    return $defaultVal;
}

/**
 * previous_month() - calculates previous month. If last day of month is selected, stick to it
 */
function previous_month($date, $no=1) {
    if ($no == 0)
        return $date;
    
    $year = format_date($date, 'Y');
    $month = format_date($date, 'n');
    $day = format_date($date, 'j');
    
    $daysInMonth = date('t', date2unix($date));
    
    $t = mktime(0, 0, 0, $month, 15, $year);
    $t = strtotime('-'.$no.' month', $t);
    
    if ($day == $daysInMonth || $day > date('t', $t)) {
        return date('Y-m-t', $t);
    } else {
        return sprintf('%s-%02d', date('Y-m', $t), $day);
    }
}

/**
 * next_month() - calculates next month. If last day of month is selected, stick to it
 */
function next_month($date, $no=1) {
    if ($no == 0)
        return $date;
    
    $year = format_date($date, 'Y');
    $month = format_date($date, 'n');
    $day = format_date($date, 'j');
    
    $daysInMonth = date('t', date2unix($date));
    
    $t = mktime(0, 0, 0, $month, 15, $year);
    $t = strtotime('+'.$no.' month', $t);
    
    if ($day == $daysInMonth || $day > date('t', $t)) {
        return date('Y-m-t', $t);
    } else {
        return sprintf('%s-%02d', date('Y-m', $t), $day);
    }
}

function next_day($date, $no=1) {
    if ($no == 0)
        return $date;
        
    $year = format_date($date, 'Y');
    $month = format_date($date, 'n');
    $day = format_date($date, 'j');
    
    $t = mktime(12, 0, 0, $month, $day, $year);
    $t = strtotime(($no<0?'-':'+') . abs($no) . ' days', $t);
    
    return date('Y-m-d', $t);
}
function previous_day($date, $no=1) {
    if ($no == 0)
        return $date;
        
    return next_day($date, $no * -1);
}


function next_week($date, $no=1) {
    return next_day($date, $no * 7);
}


function previous_week_no($year, $weekNo) {
    $weekNo--;
    
    if ($weekNo <= 0) {
        $year--;
        $weekNo = weeks_in_year($year);
    }
    
    return sprintf('%d-%02d', $year, $weekNo);
}

/**
 * 
 * @param int $dayOccurence     upwards: 1-5
 *                                  downwards for last-day: -1, -2, ...
 * 
 * @param string $dayName          mo/tu/we/th/fr/sa
 * @param int    $month            1-12
 * @param int    $year
 */
function day_in_month( $dayOccurence, $dayName, $month, $year ) {
    
    $dayname2no = array();
    $dayname2no['1'] = array('mo', 'ma');
    $dayname2no['2'] = array('tu', 'di');
    $dayname2no['3'] = array('we', 'wo');
    $dayname2no['4'] = array('th', 'do');
    $dayname2no['5'] = array('fr', 'vr');
    $dayname2no['6'] = array('sa', 'za');
    $dayname2no['7'] = array('su', 'zo');
    
    $dayN = null;
    foreach($dayname2no as $num => $names) {
        if (in_array($dayName, $names)) {
            $dayN = $num;
            break;
        }
    }
    
    if (!$dayN) {
        throw new InvalidStateException( '$dayName not found: '.$dayName );
    }
    
    
    $t = mktime( 12, 0, 0, $month, 1, $year );
    $date = date('Y-m-d', $t);
    
    $daysInMonth = date('t', $t);
    $occurrenceCount = 0;
    
    
    if ($dayOccurence > 0) {
        for($x=0; $x < $daysInMonth; $x++) {
            $d = next_day($date, $x);
            
            if (format_date($d, 'N') == $dayN) {
                $occurrenceCount++;
                if ($dayOccurence == $occurrenceCount)
                    return $d;
            }
        }
    }
    else {
        $date = format_date( $date, 'Y-m-t' );
        for($x=0; $x < $daysInMonth; $x++) {
            $d = previous_day($date, $x);
            
            if (format_date($d, 'N') == $dayN) {
                $occurrenceCount++;
                if (abs($dayOccurence) == $occurrenceCount)
                    return $d;
            }
        }
    }
    
    return null;
}


function next_week_no($year, $weekNo) {
    $weekNo++;
    
    if ($weekNo > weeks_in_year( $year )) {
        $weekNo = 1;
        $year++;
    }
    
    return sprintf('%d-%02d', $year, $weekNo);
}

function date_add_hours( $date, $hours, $format='Y-m-d H:i:s' ) {
    $dt = new DateTime( $date, new DateTimeZone(date_default_timezone_get()) );
    if ($hours > 0) {
        $dt->add(new DateInterval('PT'.$hours.'H'));
    }
    else if ($hours < 0) {
        $dt->sub(new DateInterval('PT'.abs($hours).'H'));
    }
    
    return $dt->format( $format );
}



function weeks_in_year($year, $timezone=null) {
    if ($timezone === null) {
        $timezone = date_default_timezone_get();
    }
    
    $dt = new DateTime($year . '-12-30', new DateTimeZone($timezone));
    
    // ISO-8601 specification, 28 dec always last week
    $dt->setDate($year, 12, 28);
    
    return $dt->format('W');
}

/**
 * week_list() - returns array with weeks, year & start date
 * 
 * week_list(2019) returns,
 *  array(
 *      array('weekno' => '01'..., 'year' => '2018', 'monday' => '2018-12-31'),
 *      ...
 *      array('weekno' => '52'..., 'year' => '2019', 'monday' => '2018-12-23'),
 *  )
 */
function week_list($year, $timezone='Europe/Amsterdam') {
    $r = array();
    
    $dt = new DateTime(($year-1).'-12-23', new DateTimeZone($timezone));
    
    while ($dt->format('W') > 50) {
        $dt->modify('+1 day');
    }
    
    $x=0;
    $blnWeek1Set = false;
    while(true) {
        // monday?
        if ($dt->format('N') == 1) {
            $weekno = $dt->format('Y-W');
            
            // another 01-week? => next year => break
            if ($dt->format('W') == '01' && $blnWeek1Set) {
                break;
            }
            
            // mark week 01 as set
            if ($dt->format('W') == '01') {
                $blnWeek1Set = true;
            }
            
            $r[] = array(
                'weekno' => (int)$dt->format('W'),
                'year' => $dt->format('o'),
                'monday' => $dt->format('Y-m-d')
            );
        }
        
        $dt->modify('+1 day');
    }
    
    return $r;
}

/**
 * week_diff() - returnweeks between start & end
 */
function week_diff($p_startYear, $p_startWeek, $p_endYear, $p_endWeek) {

    $startYear = (int)$p_startYear;
    $startWeek = (int)$p_startWeek;
    $endYear = (int) $p_endYear;
    $endWeek = (int)$p_endWeek;
    
    if ($startYear == $endYear && $startWeek == $endWeek) {
        return 0;
    }
    
    $negative = false;
    if ($startYear > $endYear || ($startYear == $endYear && $startWeek > $endWeek)) {
        $negative = true;
    }
    
    $startDate = null;
    $weeklistStart = week_list($startYear);
    if ($startWeek < 1 || $startWeek > count($weeklistStart)) {
        throw new InvalidStateException('Invalid start week/year ('.$p_startWeek.'/'.$p_startYear.')');
    }
    $startDate = $weeklistStart[$startWeek-1]['monday'];
    
    $endDate = null;
    $weeklistEnd = week_list($endYear);
    if ($endWeek < 1 || $endWeek > count($weeklistEnd)) {
        throw new InvalidStateException('Invalid end week/year ('.$p_endWeek.'/'.$p_endYear.')');
    }
    $endDate = $weeklistEnd[$endWeek-1]['monday'];
    
    $weeknos = 0;
    if ($endYear > $startYear) {
        // add weeks for this year
        $weeknos += (weeks_in_year($startYear) - $startWeek);
        
        // add weeks for years in between
        for($x=$startYear+1; $x < $endYear; $x++) {
            $weeknos += weeks_in_year($x);
        }
        
        // add weeks for last year
        $weeknos += $endWeek;
    }
    if ($endYear < $startYear) {
        // subtract weeks this year
        $weeknos -= $startWeek;
        
        for($x=$startYear-1; $x > $endYear; $x--) {
            $weeknos -= weeks_in_year($x);
        }
        
        $weeknos -= (weeks_in_year($endYear) - $endWeek);
    }
    if ($startYear == $endYear) {
        return $endWeek - $startWeek;
    }
    
    
    return $weeknos;
}


function week2date($year, $week) {
    $dt = new DateTime($year.'-01-01', new DateTimeZone(date_default_timezone_get()));
    $dt->setISODate($year, $week);
    
    return $dt->format('Y-m-d');
}

/**
 * week2date_day() - return day in given week
 *                 - weeks are from monday till sunday
 *                 @param dayNo
 *                 - 1 = monday
 *                 - 2 = tuesday
 *                 - 3 = wednesday
 *                 - 4 = thursday
 *                 - 5 = friday
 *                 - 6 = saturday
 *                 - 7 = sunday
 */
function week2date_day1($year, $week, $dayNo) {
    
    $date = week2date( $year, $week );
    list ($y, $m, $d) = explode('-', $date);
    
    $dt = new DateTime();
    $dt->setDate($y, $m, $d);
    
    $diff_days = 0;
    if ($dayNo > $dt->format('N')) {
        $diff_days = $dayNo - $dt->format('N');
    }
    else if ($dayNo < $dt->format('N')) {
        $diff_days = $dayNo - $dt->format('N');
    }
    if ($diff_days != 0) {
        $di = new DateInterval('P1D');
        $di->d = $diff_days;
        $dt->add( $di );
    }
    
    return $dt->format('Y-m-d');
}




function valid_date($str) {
    if ($str == '0000-00-00' || $str == false)
        return false;
    
    if (preg_match('/^\\d\\d-\\d\\d-\\d\\d\\d\\d$/', $str)) {
        return true;
    }
    if (preg_match('/^\\d\\d\\d\\d-\\d\\d-\\d\\d$/', $str)) {
        return true;
    }
    
    return false;
}

function valid_datetime($str) {
    if ($str === null)
        return false;
    
    if (preg_match('/^\\d{4}-\\d{2}-\\d{2}T\\d{2}:\\d{2}:\\d{2}/', $str)) {
        try {
            // try to parse DateTime
            $dt = new DateTime( $str );
            
            return true;
        } catch (\Exception $ex) {
            return false;
        }
    }
    
    if ($str == '0000-00-00 00:00:00')
        return false;
    
    if (preg_match('/^\\d\\d-\\d\\d-\\d\\d\\d\\d \\d\\d:\\d\\d:\\d\\d$/', $str)) {
        return true;
    }
    if (preg_match('/^\\d\\d\\d\\d-\\d\\d-\\d\\d \\d\\d:\\d\\d:\\d\\d$/', $str)) {
        return true;
    }

    if (preg_match('/^\\d\\d\\d\\d-\\d\\d-\\d\\d \\d\\d:\\d\\d$/', $str)) {
        return true;
    }
    if (preg_match('/^\\d\\d-\\d\\d-\\d\\d\\d\\d \\d\\d:\\d\\d$/', $str)) {
        return true;
    }
    
    return false;
}

function valid_time($str) {
    if ($str && preg_match('/^\\d\\d:\\d\\d:\\d\\d$/', $str)) {
        return true;
    }
    if ($str && preg_match('/^\\d\\d:\\d\\d$/', $str)) {
        return true;
    }
    
    return false;
}
function validate_email($email)
{
    if (is_string($email) == false)
        $email = '';
    
    $isValid = true;
    $atIndex = strrpos($email, "@");

    if (is_bool($atIndex) && ! $atIndex) {
        $isValid = false;
    } else {
        $domain = substr($email, $atIndex + 1);
        $local = substr($email, 0, $atIndex);
        $localLen = strlen($local);
        $domainLen = strlen($domain);
        if ($localLen < 1 || $localLen > 64) {
            // local part length exceeded
            $isValid = false;
        } else if ($domainLen < 1 || $domainLen > 255) {
            // domain part length exceeded
            $isValid = false;
        } else if ($local[0] == '.' || $local[$localLen - 1] == '.') {
            // local part starts or ends with '.'
            $isValid = false;
        } else if (preg_match('/\\.\\./', $local)) {
            // local part has two consecutive dots
            $isValid = false;
        } else if (! preg_match('/^[A-Za-z0-9\\-\\.]+$/', $domain)) {
            // character not valid in domain part
            $isValid = false;
        } else if (preg_match('/\\.\\./', $domain)) {
            // domain part has two consecutive dots
            $isValid = false;
        } else if (! preg_match('/^(\\\\.|[A-Za-z0-9!#%&`_=\\/$\'*+?^{}|~.-])+$/', str_replace("\\\\", "", $local))) {
            // character not valid in local part unless
            // local part is quoted
            if (! preg_match('/^"(\\\\"|[^"])+"$/', str_replace("\\\\", "", $local))) {
                $isValid = false;
            }
        }
        else if ( strpos($domain, '.') === false ) {
            $isValid = false;
        }
        // if ($isValid && !(checkdnsrr($domain,"MX") || checkdnsrr($domain,"A")))
        // {
        // // domain not found in DNS
        // $isValid = false;
        // }
    }
    return $isValid;
}


function mask_email($email) {
    $str = $email;
    
    $len = strlen($str);
    
    $pos_last_dot = strrpos($email, '.');
    
    for($x=0; $x < $len; $x++) {
        // show first char
        if ($x == 0) {
            
        }
        // show domain extension
        else if ($pos_last_dot !== false && $x >= $pos_last_dot-1) {
            
        }
        // show char before @ when name is >=5, show '@', show char after '@'
        else if (($x+1 < $len && $str[$x+1] == '@' && $x >= 5) || $str[$x] == '@' || ($x-1 >= 0 && $str[$x-1] == '@')) {
            
        } else {
            $str[$x] = '*';
        }
    }
    
    return $str;
}


function timediff_minuts($start, $end) {
    $dt1 = new DateTime($start);
    $dt2 = new DateTime($end);
    $diff = $dt1->diff($dt2);
    
    $minuts = ($diff->days * 24 * 60) + ($diff->h*60) + $diff->i;
    
    return $minuts;
}


function slugify($str) {
    // nothing to slugify? :)
    if ($str == null)
        return '';
    
    $str = lcfirst($str);
    $str = str_replace(['[', ']'], '-', $str);
    $str = preg_replace_callback('/[A-Z]/', function($str) { return '-'.strtolower($str[0]); }, $str);
    
    $str = strtolower($str);
    $str = trim($str);
    $str = str_replace('\\', '-', $str);
    $str = preg_replace('/[^a-z0-9 \\-\\_]/', '', $str);
    $str = str_replace(' ', '-', $str);
    $str = str_replace('_', '-', $str);
    $str = preg_replace('/\\-+/', '-', $str);
    $str = preg_replace('/^\\-+/', '', $str);
    $str = preg_replace('/\\-+$/', '', $str);
    
    return $str;
}

function limit_text($str, $maxlen, $suf='...') {
    if (strlen($str) < $maxlen) {
        return $str;
    }
    
    $str = substr($str, 0, $maxlen-(strlen($suf))) . $suf;
    
    return $str;
}

/**
 * to_php_string() - escapes a string for insertion in php-code
 * 
 * TODO: check this for security issues!! watch it before using this
 */
function to_php_string($str) {
    $str = (string)$str;
    $str = str_replace("\n", "\\n", $str);
    $str = str_replace('"', '\"', $str);
    $str = '"'.$str.'"';
    return $str;
}


function endsWith($haystack, $val) {
    $p = strrpos($haystack, $val);
    
    if ($p === false)
        return false;
    
    if ($p === (strlen($haystack) - strlen($val))) {
        return true;
    } else {
        return false;
    }
}

function endsiWith($haystack, $val) {
    $haystack = strtolower($haystack);
    $val = strtolower($val);
    
    return endsWith($haystack, $val);
}


function filterPrefixes( $str, $prefixes=array() ) {
    do {
        $matchFound = false;
        foreach($prefixes as $spf) {
            if (strpos($str, $spf) === 0) {
                $str = substr($str, strlen($spf));
                $str = trim($str);
                
                $matchFound = true;
                break;
            }
        }
    } while ($matchFound == true);
    
    return $str;
}



if (function_exists('mb_trim') == false) {
    function mb_trim($str) {
        return preg_replace("/(^\s+)|(\s+$)/u", "", $str);
    }
}

function trim_array( $arr) {
    foreach( $arr as $key => $v ) {
        if (is_string( $v ))
            $arr[$key] = trim( $v );
    }
    
    return $arr;
}


function guidv4()
{
    if (function_exists('com_create_guid') === true)
        return trim(com_create_guid(), '{}');
        
    $data = openssl_random_pseudo_bytes(16);
    $data[6] = chr(ord($data[6]) & 0x0f | 0x40); // set version to 0100
    $data[8] = chr(ord($data[8]) & 0x3f | 0x80); // set bits 6-7 to 10
    return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
}


function valid_regexp($pattern) {
    @preg_match('/'.$pattern.'/', '');
    
    return preg_last_error() === PREG_NO_ERROR ? true : false;
}


function minutes_between( $start, $end, $absolute=true ) {
    
    $dts = new DateTime( $start );
    $dte = new DateTime( $end );
    
    $diff = $dts->diff( $dte );
    $mins = 0;
    $mins += $diff->days * 24 * 60;
    $mins += $diff->h * 60;
    $mins += $diff->i;
    
    if ($absolute && $mins < 0)
        $mins = abs($mins);
    
    return $mins;
}


function days_between($start, $end, $absolute=true) {
    $s = date2unix($start);
    $e = date2unix($end);
    
    $dt1 = new DateTime(date('Y-m-d', $s));
    $dt2 = new DateTime(date('Y-m-d', $e));
    
    if ($absolute) {
        return $dt1->diff($dt2, true)->days;
    } else {
        $diff = $dt1->diff($dt2, false);
        
        return $diff->days * ($diff->invert?-1:1);
    }
}

function months_between($start, $end) {
    $s = date2unix($start);
    $e = date2unix($end);
    
    $dt1 = new DateTime(date('Y-m-d', $s));
    $dt2 = new DateTime(date('Y-m-d', $e));
    
    return $dt1->diff($dt2, true)->m;
}


function hours_between( $time1, $time2 ) {
    if ( valid_time($time1) == false || valid_time($time2) == false )
        return null;
    
    $tok1 = explode(':', $time1);
    $tok2 = explode(':', $time2);
    
    $h1 = (int)$tok1[0];
    $m1 = (int)$tok1[1];
    
    $h2 = (int)$tok2[0];
    $m2 = (int)$tok2[1];
    
    
    $mindiff = ($h2 - $h1) * 60;
    $mindiff -= $m1;
    $mindiff += $m2;
    
    return $mindiff;
}



function toolbox_mime_content_type($filename) {
    $mime_types = array(
        
        'txt'  => 'text/plain',
        'htm'  => 'text/html',
        'html' => 'text/html',
        'php'  => 'text/html',
        'css'  => 'text/css',
        'js'   => 'application/javascript',
        'json' => 'application/json',
        'xml'  => 'application/xml',
        'swf'  => 'application/x-shockwave-flash',
        'flv'  => 'video/x-flv',
        
        // images
        'png'  => 'image/png',
        'jpe'  => 'image/jpeg',
        'jpeg' => 'image/jpeg',
        'jpg'  => 'image/jpeg',
        'gif'  => 'image/gif',
        'bmp'  => 'image/bmp',
        'ico'  => 'image/vnd.microsoft.icon',
        'tiff' => 'image/tiff',
        'tif'  => 'image/tiff',
        'svg'  => 'image/svg+xml',
        'svgz' => 'image/svg+xml',
        
        // archives
        'zip' => 'application/zip',
        'rar' => 'application/x-rar-compressed',
        'exe' => 'application/x-msdownload',
        'msi' => 'application/x-msdownload',
        'cab' => 'application/vnd.ms-cab-compressed',
        
        // audio/video
        'wav' => 'audio/wav',
        'ogg '=> 'audio/ogg',
        'mp3' => 'audio/mpeg',
        'qt'  => 'video/quicktime',
        'mov' => 'video/quicktime',
        
        // adobe
        'pdf' => 'application/pdf',
        'psd' => 'image/vnd.adobe.photoshop',
        'ai'  => 'application/postscript',
        'eps' => 'application/postscript',
        'ps'  => 'application/postscript',
        
        // ms office
        'doc' => 'application/msword',
        'rtf' => 'application/rtf',
        'xls' => 'application/vnd.ms-excel',
        'ppt' => 'application/vnd.ms-powerpoint',
        
        // open office
        'odt' => 'application/vnd.oasis.opendocument.text',
        'ods' => 'application/vnd.oasis.opendocument.spreadsheet',
    );
    
    $p = strrpos($filename, '.');
    if ($p === false) {
        return 'application/octet-stream';
    }
    
    $ext = strtolower(substr($filename, $p+1));
    if (array_key_exists($ext, $mime_types)) {
        return $mime_types[$ext];
    }
    elseif (function_exists('finfo_open')) {
        $finfo = finfo_open(FILEINFO_MIME);
        $mimetype = finfo_file($finfo, $filename);
        finfo_close($finfo);
        return $mimetype;
    }
    else {
        return 'application/octet-stream';
    }
}
if(!function_exists('mime_content_type')) {
    function mime_content_type($filename) {
        return toolbox_mime_content_type( $filename );
    }
}

/**
 * crc32_int32() - on 64-bits systems, php always uses int64's. crc32_int32 returns
 *                 the 32-bits value (in a 64-bits int...;)
 */
function crc32_int32($str) {
    $i = crc32($str);
    
    // credits to comment jian @ https://www.php.net/manual/en/function.crc32.php
    if ($i > 2147483647){
        return $i - 2147483647 * 2 - 2;
    } else {
        return $i;
    }
}




function cleanup_string($string) {
    $co = array('á', 'â', 'à', 'ä', 'é', 'ê', 'è', 'ë', 'í', 'î', 'ì', 'ï', 'ó', 'ô', 'ò', 'ö', 'ú', 'û', 'ù', 'ü', 'Á', 'Â', 'À', 'Ä', 'É', 'Ê', 'È', 'Ë', 'Í', 'Î', 'Ì', 'Ï', 'Ó', 'Ô', 'Ò', 'Ö', 'Ú', 'Û', 'Ù', 'Ü');
    $cn = array('a', 'a', 'a', 'a', 'e', 'e', 'e', 'e', 'i', 'i', 'i', 'i', 'o', 'o', 'o', 'o', 'u', 'u', 'u', 'u', 'A', 'A', 'A', 'A', 'E', 'E', 'E', 'E', 'I', 'I', 'I', 'I', 'O', 'O', 'O', 'O', 'U', 'U', 'U', 'U');
    
    $string = str_replace($co, $cn, $string);
    
    $str = '';
    for($x=0; $x < strlen($string); $x++) {
        $c = ord($string[$x]);
        if ($c == 9 || $c == 10 || $c == 13 || ($c >= 32 && $c <= 126)) {
            $str .= $string[$x];
        }
    }
    
    return $str;
}


function excel_date2daycount( $date ) {
    if (valid_date($date) == false)
        return null;
    
    $d = format_date($date, 'Y-m-d');
    
    return days_between('1900-12-30', $d, false )+1;
}

function excel_daycount2date( $dayCount ) {
    // 25569-days = difference start unix timestamp
    $t = ($dayCount - 25569) * 86400;
    
    return date('Y-m-d', $t);
}



function in_array_callback( $needle, $array, $callback ) {
    foreach($array as $a) {
        if ( $callback( $needle, $a ) == true ) {
            return true;
        }
    }
    
    return false;
}



